package ar.edu.unju.fi.jpaoverview.service;

import ar.edu.unju.fi.jpaoverview.domain.Rol;
import ar.edu.unju.fi.jpaoverview.domain.Usuario;
import ar.edu.unju.fi.jpaoverview.repository.RolRepository;
import ar.edu.unju.fi.jpaoverview.repository.UsuarioRepository;
import jakarta.validation.Valid;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;

/**
 * Ejemplo básico del servicio para Usuario
 * @author José Zapana
 */
@Service
public class UsuarioService {

    private final UsuarioRepository usuarioRepository;
    private final RolRepository rolRepository;

    /**
     * Constructor del servicio donde se inyectan los repository
     * @param usuarioRepository
     * @param rolRepository
     */
    public UsuarioService(UsuarioRepository usuarioRepository, RolRepository rolRepository) {
        this.usuarioRepository = usuarioRepository;
        this.rolRepository = rolRepository;
    }

    /**
     * Crear un usuario
     * @param usuario
     * @return
     */
    @Transactional
    public Usuario crear(@Valid Usuario usuario) {
        usuarioRepository.findByEmail(usuario.getEmail()).ifPresent(ex -> {
            throw new IllegalArgumentException("Email ya registrado: " + usuario.getEmail());
        });
        usuarioRepository.findByUsername(usuario.getUsername()).ifPresent(ex -> {
            throw new IllegalArgumentException("Username ya registrado: " + usuario.getUsername());
        });

        if (usuario.getFechaAlta() == null) {
            usuario.setFechaAlta(LocalDateTime.now());
        }

        // FIXME asegurar rol persistido debería estar modularizado
        Rol rol = usuario.getRol();
        if (rol.getId() == null) {
            final Rol rolRef = rol;
            rol = rolRepository.findByDescripcionIgnoreCase(rol.getDescripcion())
                    .orElseGet(() -> rolRepository.save(rolRef));
        }
        usuario.setRol(rol);

        return usuarioRepository.save(usuario);
    }

    /**
     * Listar todos los usuarios
     * @return
     */
    @Transactional(readOnly = true)
    public List<Usuario> listar() {
        return usuarioRepository.findAll();
    }

    /**
     * Buscar usuario por username
     * @param username
     * @return
     */
    @Transactional(readOnly = true)
    public Usuario buscarPorUsername(String username) {
        return usuarioRepository.findByUsername(username)
                .orElseThrow(() -> new IllegalArgumentException("No existe usuario: " + username));
    }
}
