package ar.edu.unju.fi.jpaoverview.controller;

import ar.edu.unju.fi.jpaoverview.controller.dto.MensajeError;
import ar.edu.unju.fi.jpaoverview.dto.UsuarioDTO;
import ar.edu.unju.fi.jpaoverview.service.UsuarioService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.ArraySchema;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.net.URI;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@RestController
@RequestMapping("/api/usuarios")
@Tag(name = "Usuarios", description = "Operaciones sobre usuarios")
public class UsuarioController {

    private final UsuarioService usuarioService;

    public UsuarioController(UsuarioService usuarioService) {
        this.usuarioService = usuarioService;
    }

    /**
     * Crear usuario
     * @param dto
     * @return
     */
    @Operation(
            summary = "Crear un nuevo usuario",
            description = "Crea un usuario validando email y username únicos y persistiendo/creando el rol si corresponde",
            responses = {
                    @ApiResponse(
                            responseCode = "201",
                            description = "Usuario creado",
                            content = @Content(mediaType = "application/json", schema = @Schema(implementation = UsuarioDTO.class))
                    ),
                    @ApiResponse(
                            responseCode = "400",
                            description = "Error de validación o negocio",
                            content = @Content(mediaType = "application/json", schema = @Schema(implementation = MensajeError.class))
                    )
            }
    )
    @PostMapping
    public ResponseEntity<UsuarioDTO> crear(@Valid @RequestBody UsuarioDTO dto) {
        log.info("Creando un nuevo usuario");
        UsuarioDTO creado = usuarioService.crear(dto);
        return ResponseEntity.created(URI.create("/api/usuarios/" + creado.getUsername())).body(creado);
    }

    /**
     * Listar usuarios
     * @return
     */
    @Operation(summary = "Listar usuarios",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Listado de usuarios",
                            content = @Content(mediaType = "application/json",
                                    array = @ArraySchema(schema = @Schema(implementation = UsuarioDTO.class))))
            })
    @GetMapping
    public ResponseEntity<List<UsuarioDTO>> listar() {
        log.info("Listando usuarios...");
        List<UsuarioDTO> lista = usuarioService.listar();
        return ResponseEntity.ok(lista);
    }

    /**
     * Buscar usuario por username
     * @param username
     * @return
     */
    @Operation(
            summary = "Buscar usuario por username",
            description = "Devuelve un usuario si el username coincide exactamente",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Usuario encontrado",
                            content = @Content(mediaType = "application/json", schema = @Schema(implementation = UsuarioDTO.class))),
                    @ApiResponse(responseCode = "404", description = "Usuario no encontrado",
                            content = @Content(mediaType = "application/json",
                                    schema = @Schema(implementation = MensajeError.class)))
            }
    )
    @GetMapping(params = "username")
    public ResponseEntity<?> buscarPorUsernameParam(
            @Parameter(description = "Nombre de usuario", required = true)
            @RequestParam String username) {
        log.info("Buscando usuario por parámetro username: {}", username);
        UsuarioDTO dto = usuarioService.buscarPorUsername(username);
        if (dto != null) {
            return ResponseEntity.ok(dto);
        } else {
            Map<String, String> error = new HashMap<>();
            error.put("mensaje", "No se encontró ningún usuario con username: " + username);
            return ResponseEntity.status(HttpStatus.NOT_FOUND).body(error);
        }
    }
}
