package ar.edu.unju.fi.jpaoverview;

import ar.edu.unju.fi.jpaoverview.domain.Rol;
import ar.edu.unju.fi.jpaoverview.domain.Usuario;
import ar.edu.unju.fi.jpaoverview.dto.UsuarioDTO;
import ar.edu.unju.fi.jpaoverview.repository.RolRepository;
import ar.edu.unju.fi.jpaoverview.repository.UsuarioRepository;
import ar.edu.unju.fi.jpaoverview.service.UsuarioService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.*;
import org.mockito.junit.jupiter.MockitoExtension;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

import static org.assertj.core.api.Assertions.*;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
class UsuarioServiceTest {

    @Mock
    UsuarioRepository usuarios;

    @Mock
    RolRepository roles;

    @InjectMocks
    UsuarioService service;

    Rol rolUser;

    @BeforeEach
    void setUp() {
        rolUser = Rol.builder().id(1L).descripcion("USER").build();
    }

    @Test
    void crear_usuarioNuevo_guardaYDevuelve_conFechaAltaYRolPersistido() {
        // given: email/username no existen y rol “USER” existe en BD
        when(usuarios.findByEmail("ana@ejemplo.com")).thenReturn(Optional.empty());
        when(usuarios.findByUsername("ana")).thenReturn(Optional.empty());
        when(roles.findByDescripcionIgnoreCase("USER")).thenReturn(Optional.of(rolUser));

        // el save del usuario devuelve uno con id
        ArgumentCaptor<Usuario> captor = ArgumentCaptor.forClass(Usuario.class);
        when(usuarios.save(any(Usuario.class))).thenAnswer(inv -> {
            Usuario u = inv.getArgument(0);
            u.setId(100L);
            return u;
        });

        UsuarioDTO entrada = UsuarioDTO.builder()
                .email("ana@ejemplo.com")
                .nombre("Ana")
                .username("ana")
                .password("secreto123")
                .rol("USER")
                .build();

        // when
        UsuarioDTO guardado = service.crear(entrada);

        // then
        verify(usuarios).findByEmail("ana@ejemplo.com");
        verify(usuarios).findByUsername("ana");
        verify(roles).findByDescripcionIgnoreCase("USER");
        verify(usuarios).save(captor.capture());

        Usuario pasadoARepo = captor.getValue();
        assertThat(pasadoARepo.getRol()).isSameAs(rolUser);
        assertThat(pasadoARepo.getFechaAlta()).isNotNull();            // se setea si venía null
        assertThat(guardado.getId()).isEqualTo(100L);
        assertThat(guardado.getPassword()).isNull(); // no se expone password en DTO de salida
        assertThat(guardado.getRol()).isEqualTo("USER");
    }

    @Test
    void crear_emailDuplicado_lanzaIllegalArgumentException() {
        when(usuarios.findByEmail("dup@ejemplo.com"))
                .thenReturn(Optional.of(Usuario.builder().id(9L).email("dup@ejemplo.com").build()));

        UsuarioDTO entrada = UsuarioDTO.builder()
                .email("dup@ejemplo.com")
                .nombre("Test")
                .username("test")
                .password("password123")
                .rol("USER")
                .build();

        assertThatThrownBy(() -> service.crear(entrada))
                .isInstanceOf(IllegalArgumentException.class)
                .hasMessageContaining("Email ya registrado");

        verify(usuarios, never()).save(any());
    }

    @Test
    void crear_usernameDuplicado_lanzaIllegalArgumentException() {
        when(usuarios.findByEmail("ok@ejemplo.com")).thenReturn(Optional.empty());
        when(usuarios.findByUsername("existente"))
                .thenReturn(Optional.of(Usuario.builder().id(7L).username("existente").build()));

        UsuarioDTO entrada = UsuarioDTO.builder()
                .email("ok@ejemplo.com")
                .nombre("Ok")
                .username("existente")
                .password("password123")
                .rol("USER")
                .build();

        assertThatThrownBy(() -> service.crear(entrada))
                .isInstanceOf(IllegalArgumentException.class)
                .hasMessageContaining("Username ya registrado");

        verify(usuarios, never()).save(any());
    }

    @Test
    void crear_rolNoExistente_loCreaYLoAsocia() {
        // no existe email/username
        when(usuarios.findByEmail(anyString())).thenReturn(Optional.empty());
        when(usuarios.findByUsername(anyString())).thenReturn(Optional.empty());

        // el rol pasado no existe por descripción -> se guarda
        when(roles.findByDescripcionIgnoreCase("OPERADOR")).thenReturn(Optional.empty());
        Rol rolPersistido = Rol.builder().id(5L).descripcion("OPERADOR").build();
        when(roles.save(argThat(r -> r.getId() == null && "OPERADOR".equals(r.getDescripcion()))))
                .thenReturn(rolPersistido);

        when(usuarios.save(any(Usuario.class))).thenAnswer(inv -> {
            Usuario u = inv.getArgument(0);
            u.setId(11L);
            return u;
        });

        UsuarioDTO entrada = UsuarioDTO.builder()
                .email("op@ejemplo.com")
                .nombre("Oper")
                .username("oper")
                .password("password123")
                .rol("OPERADOR")
                .build();

        UsuarioDTO guardado = service.crear(entrada);

        assertThat(guardado.getId()).isEqualTo(11L);
        assertThat(guardado.getRol()).isEqualTo("OPERADOR");
        verify(roles).save(argThat(r -> "OPERADOR".equals(r.getDescripcion())));
    }

    @Test
    void listar_ok() {
        when(usuarios.findAll()).thenReturn(List.of(
                Usuario.builder().id(1L).username("a").email("a@x.com").rol(rolUser).fechaAlta(LocalDateTime.now()).build(),
                Usuario.builder().id(2L).username("b").email("b@x.com").rol(rolUser).fechaAlta(LocalDateTime.now()).build()
        ));

        var lista = service.listar();
        assertThat(lista).hasSize(2);
        assertThat(lista.get(0).getRol()).isEqualTo("USER");
        verify(usuarios).findAll();
    }

    @Test
    void buscarPorUsername_ok() {
        when(usuarios.findByUsername("ana")).thenReturn(Optional.of(
                Usuario.builder().id(1L).username("ana").email("ana@x.com").rol(rolUser).fechaAlta(LocalDateTime.now()).build()
        ));

        var u = service.buscarPorUsername("ana");
        assertThat(u.getEmail()).isEqualTo("ana@x.com");
        assertThat(u.getRol()).isEqualTo("USER");
        verify(usuarios).findByUsername("ana");
    }

    @Test
    void buscarPorUsername_inexistente_lanzaIllegalArgumentException() {
        when(usuarios.findByUsername("noexiste")).thenReturn(Optional.empty());
        assertThatThrownBy(() -> service.buscarPorUsername("noexiste"))
                .isInstanceOf(IllegalArgumentException.class)
                .hasMessageContaining("No existe usuario");
    }
}
