package ar.edu.unju.fi.jpaoverview.service;

import ar.edu.unju.fi.jpaoverview.domain.Rol;
import ar.edu.unju.fi.jpaoverview.domain.Usuario;
import ar.edu.unju.fi.jpaoverview.dto.UsuarioDTO;
import ar.edu.unju.fi.jpaoverview.repository.RolRepository;
import ar.edu.unju.fi.jpaoverview.repository.UsuarioRepository;
import jakarta.validation.Valid;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Ejemplo básico del servicio para Usuario
 * @author José Zapana
 */
@Service
@Slf4j
public class UsuarioService {

    private final UsuarioRepository usuarioRepository;
    private final RolRepository rolRepository;

    public UsuarioService(UsuarioRepository usuarioRepository, RolRepository rolRepository) {
        this.usuarioRepository = usuarioRepository;
        this.rolRepository = rolRepository;
    }

    // Mappers
    private UsuarioDTO toDto(Usuario u) {
        if (u == null) return null;
        return UsuarioDTO.builder()
                .id(u.getId())
                .email(u.getEmail())
                .nombre(u.getNombre())
                .username(u.getUsername())
                // Por seguridad no se expone password en salida
                .password(null)
                .fechaAlta(u.getFechaAlta())
                .dni(u.getDni())
                .rol(u.getRol() != null ? u.getRol().getDescripcion() : null)
                .build();
    }

    private Usuario toEntity(UsuarioDTO dto) {
        if (dto == null) return null;
        Rol rol = rolRepository.findByDescripcionIgnoreCase(dto.getRol())
                .orElseGet(() -> Rol.builder().descripcion(dto.getRol()).build());
        return Usuario.builder()
                .id(dto.getId())
                .email(dto.getEmail())
                .nombre(dto.getNombre())
                .username(dto.getUsername())
                .password(dto.getPassword())
                .fechaAlta(dto.getFechaAlta())
                .dni(dto.getDni())
                .rol(rol)
                .build();
    }

    /**
     * Crear un usuario
     */
    @Transactional
    public UsuarioDTO crear(@Valid UsuarioDTO usuarioDto) {
        log.info("Inicio crear usuario en el service...");
        usuarioRepository.findByEmail(usuarioDto.getEmail()).ifPresent(ex -> {
            log.error("Email ya registrado {}", ex.getEmail());
            throw new IllegalArgumentException("Email ya registrado: " + usuarioDto.getEmail());
        });
        usuarioRepository.findByUsername(usuarioDto.getUsername()).ifPresent(ex -> {
            log.error("Usuario ya registrado {}", ex.getUsername());
            throw new IllegalArgumentException("Username ya registrado: " + usuarioDto.getUsername());
        });

        if (usuarioDto.getFechaAlta() == null) {
            usuarioDto.setFechaAlta(LocalDateTime.now());
        }

        // asegurar rol persistido
        Rol rol = rolRepository.findByDescripcionIgnoreCase(usuarioDto.getRol())
                .orElseGet(() -> rolRepository.save(Rol.builder().descripcion(usuarioDto.getRol()).build()));

        Usuario usuario = toEntity(usuarioDto);
        usuario.setRol(rol);

        Usuario guardado = usuarioRepository.save(usuario);
        return toDto(guardado);
    }

    /**
     * Listar todos los usuarios
     */
    @Transactional(readOnly = true)
    public List<UsuarioDTO> listar() {
        return usuarioRepository.findAll().stream().map(this::toDto).collect(Collectors.toList());
    }

    /**
     * Buscar usuario por username
     */
//    @Transactional(readOnly = true)
//    public UsuarioDTO buscarPorUsername(String username) {
//        Usuario u = usuarioRepository.findByUsername(username)
//                .orElseThrow(() -> new IllegalArgumentException("No existe usuario: " + username));
//        return toDto(u);
//    }

    @Transactional(readOnly = true)
    public UsuarioDTO buscarPorUsername(String username) {
        return usuarioRepository.findByUsername(username)
                .map(this::toDto)
                .orElse(null);
    }
}
